"""Nagios check implementation.

Based on the specification from
https://assets.nagios.com/downloads/nagioscore/docs/nagioscore/3/en/pluginapi.html
"""
# pylint: disable=logging-fstring-interpolation

import collections
import enum
import logging
import subprocess  # nosec

DEFAULT_TIMEOUT = 10  # In seconds.


class NagiosCode(enum.IntEnum):
    """Enum for agreed upon exit codes for a Nagios plugin."""

    OK = 0
    WARNING = 1
    CRITICAL = 2
    UNKNOWN = 3


PerfData = collections.namedtuple(
    "PerfData", ["name", "value", "unit", "warning", "critical", "min", "max"]
)


class Check:
    # pylint: disable=invalid-name
    """Nagios check using a plugin."""

    Command = None
    Arguments = []
    ExitCode = None
    Output = None
    PerfData = []
    LongOutput = None
    AdditionalOutput = None
    stderr = None
    _stdout = None

    def __init__(self, command, args=None):
        if not command:
            raise ValueError("Command is empty.")
        self.Command = command
        if args:
            self.Arguments = args

    def _parse_output(self):
        """Parses the plugin output to get the output, perf data and long
        output. Can be run after the plugin has run and finished
        successfully."""
        if self.ExitCode is None:
            raise RuntimeError("Check hasn't run yet.")

    def run(self, timeout=DEFAULT_TIMEOUT):
        """Run the check (if it's hasn't run yet)."""
        if self.ExitCode is not None:
            raise Exception("Check has already run.")
        _cmd = [
            self.Command,
        ] + self.Arguments
        logging.info(f"Running command {_cmd}.")
        try:
            proc = subprocess.run(  # nosec
                [
                    self.Command,
                ]
                + self.Arguments,
                capture_output=True,
                check=False,
                text=True,
                timeout=timeout,
            )
        except FileNotFoundError:
            raise RuntimeError(f"Command {self.Command} not found.")
        except subprocess.TimeoutExpired:
            raise RuntimeError("Timeout exceeded.")
        self.ExitCode = proc.returncode
        logging.info(f"Exit code is {self.ExitCode}.")
        self._stdout = proc.stdout
        logging.info("stdout is {self.stdout}.")
        self.stderr = proc.stderr
        logging.info("stderr is {self.stderr}.")
        self._stdout = proc.stdout
        self._parse_output()
