"""Nagios check implementation.

Based on the specification from
https://assets.nagios.com/downloads/nagioscore/docs/nagioscore/3/en/pluginapi.html
"""
# pylint: disable=logging-fstring-interpolation

import collections
import enum
import logging
import subprocess  # nosec

DEFAULT_TIMEOUT = 10  # In seconds.


class NagiosCode(enum.IntEnum):
    """Enum for agreed upon exit codes for a Nagios plugin."""

    OK = 0
    WARNING = 1
    CRITICAL = 2
    UNKNOWN = 3


PerfData = collections.namedtuple(
    "PerfData", ["name", "value", "unit", "warning", "critical", "min", "max"]
)
PerfData.__new__.__defaults__ = tuple([None] * 7)


def parse_perf_data(line):  # noqa: MC0001
    """Convert a single-line string to a DataPerf tuple."""
    if len(line.splitlines()) > 1:
        raise RuntimeError("Can only parse a single line at a time.")

    fields = line.split(";")
    name, value = fields[0].split("=")

    # Clean the name, handle a quoted name.
    name = name.strip()
    if name[0] in ["'", '"']:
        name = name[1:]
    if name[-1] in ["'", '"']:
        name = name[:-1]
    name = name.strip()
    pd = {"name": name}  # pylint: disable=invalid-name

    # Split the unit and value.
    if not value[-1].isdigit():
        for i in range(len(value) - 1, 0, -1):
            if value[i].isdigit():
                break
        pd["value"], pd["unit"] = (
            float(value[: i + 1]),
            value[i + 1 :],  # noqa: E203
        )
    else:
        pd["value"] = float(value)

    # Get the remaining fields, if available.
    try:
        pd["warning"] = float(fields[1])
    except (ValueError, IndexError) as ex:
        logging.debug(str(ex))

    try:
        pd["critical"] = float(fields[2])
    except (ValueError, IndexError) as ex:
        logging.debug(str(ex))

    try:
        pd["min"] = float(fields[3])
    except (ValueError, IndexError) as ex:
        logging.debug(str(ex))

    try:
        pd["max"] = float(fields[4])
    except (ValueError, IndexError) as ex:
        logging.debug(str(ex))

    return PerfData(**pd)


class Check:
    # pylint: disable=invalid-name
    """Nagios check using a plugin."""

    Command = None
    Arguments = []
    ExitCode = None
    Output = None
    PerfData = []
    AdditionalOutput = []
    stderr = None
    _stdout = None

    def __init__(self, command, args=None):
        if not command:
            raise ValueError("Command is empty.")
        self.Command = command
        if args:
            self.Arguments = args

    def _parse_output(self):
        """Parses the plugin output to get the output, perf data and long
        output.

        Can only be run after the plugin has run and finished successfully.
        """
        if self._stdout is None or self._stdout == "" or self._stdout == "\n":
            logging.info("Empty stdout, skipping parsing.")
            return

        lines = self._stdout.splitlines()
        if not lines:
            logging.info("Empty stdout, skipping parsing.")
            return

        if "|" in lines[0]:
            p1, p2 = lines[0].split("|")
            self.Output = p1.strip()
            self.PerfData.append(parse_perf_data(p2))
        else:
            self.Output = lines[0].strip()

        if len(lines) == 1:
            return

        processing_perfdata = False
        for line in lines[1:]:
            if processing_perfdata:
                self.PerfData.append(parse_perf_data(line))
            else:
                if "|" in line:
                    self.AdditionalOutput.append(line.split("|")[0].strip())
                    processing_perfdata = True
                    self.PerfData.append(parse_perf_data(line.split("|")[1]))
                else:
                    self.AdditionalOutput.append(line)

    def run(self, timeout=DEFAULT_TIMEOUT):
        """Run the check (if it's hasn't run yet)."""
        if self.ExitCode is not None:
            raise Exception("Check has already run.")
        _cmd = [
            self.Command,
        ] + self.Arguments
        logging.info(f"Running command {_cmd}.")
        try:
            proc = subprocess.run(  # nosec
                [
                    self.Command,
                ]
                + self.Arguments,
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                check=False,
                timeout=timeout,
                universal_newlines=True,
            )
        except FileNotFoundError:
            raise RuntimeError(f"Command {self.Command} not found.")
        except subprocess.TimeoutExpired:
            raise RuntimeError("Timeout exceeded.")
        self.ExitCode = proc.returncode
        logging.info(f"Exit code is {self.ExitCode}.")
        self._stdout = proc.stdout
        logging.info("stdout is {self._stdout}.")
        self.stderr = proc.stderr
        logging.info("stderr is {self.stderr}.")
        if self.ExitCode in list(NagiosCode):
            self._parse_output()
            logging.info(f"Output is {self.Output}.")
            logging.info(f"Additional output is {self.AdditionalOutput}.")
        else:
            logging.warning(
                "Invalid plugin return code, skipping output parsing."
            )
