"""My Nagios plugin wrapper."""
# pylint: disable=logging-fstring-interpolation
__version__ = "0.1.1"

import argparse
import logging
import socket

import requests

from mnpw import nagios

HOSTNAME = socket.gethostname()


def notify(message):
    """Send a notification."""
    requests.post(
        "https://notify.shore.co.il/send", params={"message": message}
    )


def main():  # noqa: MC0001
    """Main entrypoint."""
    parser = argparse.ArgumentParser(description=__doc__)
    parser.add_argument("command", type=str, help="Plugin command")
    parser.add_argument(
        "arguments",
        type=str,
        nargs="*",
        help="Command arguments",
    )
    parser.add_argument(
        "-v", "--verbose", help="Verbose output", action="store_true"
    )
    parser.add_argument(
        "-V",
        "--version",
        action="version",
        version=f"mnpw version {__version__}",
    )
    parser.add_argument(
        "-d", "--dry-run", help="Dry-run, don't notify", action="store_true"
    )
    parser.add_argument(
        "-q",
        "--quiet",
        "--silent",
        help="No output, except for errors.",
        action="store_true",
    )
    parser.add_argument(
        "-w",
        "--warn",
        help="Notify when the check status is WARNING.",
        action="store_true",
    )
    parser.add_argument(
        "-u",
        "--unknown",
        help="Notify when the check status is UNKNOWN.",
        action="store_true",
    )
    parser.add_argument(
        "-e",
        "--errors",
        help="Notify on check errors.",
        action="store_true",
    )
    parser.add_argument(
        "-t",
        "--timeout",
        help=f"Command timeout (in seconds), defaults to {nagios.DEFAULT_TIMEOUT}",  # noqa: E501
        type=int,
        default=nagios.DEFAULT_TIMEOUT,
    )
    args = parser.parse_args()
    log_level = logging.WARNING  # Default level.
    if args.verbose and args.quiet:
        parser.error("Can't specify verbose and quiet output together.")
    if args.verbose:
        log_level = logging.INFO
    if args.quiet:
        log_level = logging.CRITICAL
    logging.basicConfig(format="%(message)s", level=log_level)

    check = nagios.Check(args.command, args.arguments)
    try:
        check.run(args.timeout)
    except Exception as ex:  # pylint: disable=broad-except
        if args.errors and not args.dry_run:
            notify(f"Check {args.command} on {HOSTNAME} failed: {ex}.")
        parser.error(str(ex))

    if check.ExitCode == nagios.NagiosCode.OK:
        logging.info("Check status is OK.")
    elif check.ExitCode == nagios.NagiosCode.WARNING:
        logging.info("Check status is WARNING.")
        if args.warn and not args.dry_run:
            notify(f"{check.Output} on {HOSTNAME}.")
    elif check.ExitCode == nagios.NagiosCode.CRITICAL:
        logging.info("Check status is CRITICAL.")
        if not args.dry_run:
            notify(f"{check.Output} on {HOSTNAME}.")
    elif check.ExitCode == nagios.NagiosCode.UNKNOWN:
        logging.info("Check status is UNKNOWN.")
        if args.unknown and not args.dry_run:
            notify(f"{check.Output} on {HOSTNAME}.")
    else:
        logging.info(
            f"Check status is invalid for a Nagios plugin ({check.ExitCode})."
        )
        if args.errors:
            notify(f"Check {args.command} on {HOSTNAME} status is invalid.")


if __name__ == "__main__":
    main()
