variable "cidr_block" {
  default     = "172.31.0.0/16"
  description = "CIDR block for the VPC."
  type        = string
}

output "cidr_block" {
  description = "CIDR block for the VPC."
  value       = var.cidr_block
}

resource "aws_vpc" "vpc" {
  cidr_block = var.cidr_block
  tags = {
    Name = local.env,
  }
}

locals {
  vpc_arn  = aws_vpc.vpc.arn
  vpc_id   = aws_vpc.vpc.id
  vpc_name = aws_vpc.vpc.tags.Name
}

output "vpc_arn" {
  description = "ARN of the VPC."
  value       = local.vpc_arn
}

output "vpc_id" {
  description = "ID of the VPC."
  value       = local.vpc_id
}

output "vpc_name" {
  description = "Name of the VPC."
  value       = local.vpc_name
}

resource "aws_internet_gateway" "gateway" {
  vpc_id = local.vpc_id
  tags = {
    Name = local.env,
  }
}

locals {
  internet_gateway_arn  = aws_internet_gateway.gateway.arn
  internet_gateway_id   = aws_internet_gateway.gateway.id
  internet_gateway_name = aws_internet_gateway.gateway.tags["Name"]
}

output "internet_gateway_arn" {
  description = "ARN of the internet gateway."
  value       = local.internet_gateway_arn
}

output "internet_gateway_id" {
  description = "ID of the internet gateway."
  value       = local.internet_gateway_id
}

output "internet_gateway_name" {
  description = "Name of the internet gateway."
  value       = local.internet_gateway_name
}

variable "subnet_count" {
  default     = 3
  description = "Number of each private and public subnets."
  type        = number
}

output "subnet_count" {
  description = "Number of each private and public subnets."
  value       = var.subnet_count
}

locals {
  az_mapping = {
    1 = "a"
    2 = "b"
    3 = "c"
    4 = "d"
    5 = "e"
    6 = "f"
    7 = "g"
    8 = "h"
    9 = "i"
  }
}

resource "aws_subnet" "private" {
  count                   = var.subnet_count
  availability_zone       = "${var.region}${local.az_mapping[count.index + 1]}"
  cidr_block              = cidrsubnet(var.cidr_block, 8, count.index)
  map_public_ip_on_launch = false
  vpc_id                  = local.vpc_id
  tags = {
    Name = "${local.env}-private-${local.az_mapping[count.index + 1]}"
    Type = "private"
  }
}

locals {
  private_subnet_arns  = aws_subnet.private.*.arn
  private_subnet_ids   = aws_subnet.private.*.id
  private_subnet_names = [for i in aws_subnet.private.*.tags : i["Name"]]
}

output "private_subnet_arns" {
  description = "List of private subnets ARNs."
  value       = local.private_subnet_arns
}

output "private_subnet_ids" {
  description = "List of private subnets IDs."
  value       = local.private_subnet_ids
}

output "private_subnet_names" {
  description = "List of private subnets names."
  value       = local.private_subnet_names
}

resource "aws_subnet" "public" {
  count                   = var.subnet_count
  availability_zone       = "${var.region}${local.az_mapping[count.index + 1]}"
  cidr_block              = cidrsubnet(var.cidr_block, 8, var.subnet_count + count.index)
  map_public_ip_on_launch = true
  vpc_id                  = local.vpc_id
  tags = {
    Name = "${local.env}-public-${local.az_mapping[count.index + 1]}"
    Type = "public"
  }
}

locals {
  public_subnet_arns  = aws_subnet.public.*.arn
  public_subnet_ids   = aws_subnet.public.*.id
  public_subnet_names = [for i in aws_subnet.public.*.tags : i["Name"]]
}

output "public_subnet_arns" {
  description = "List of public subnets ARNs."
  value       = local.public_subnet_arns
}

output "public_subnet_ids" {
  description = "List of public subnets IDs."
  value       = local.public_subnet_ids
}

output "public_subnet_names" {
  description = "List of public subnets names."
  value       = local.public_subnet_names
}

resource "aws_eip" "nat_eip" {
  count = var.subnet_count
  vpc   = true
  tags = {
    Name = "${local.env}-${local.az_mapping[count.index + 1]}"
  }
  depends_on = [
    aws_internet_gateway.gateway,
  ]
}

locals {
  nat_gateway_eip_ids   = aws_eip.nat_eip.*.id
  nat_gateway_eip_names = [for i in aws_eip.nat_eip.*.tags : i["Name"]]
}

output "nat_gateway_eip_ids" {
  description = "List of Elastic IP IDs for the NAT gateway."
  value       = local.nat_gateway_eip_ids
}

output "nat_gateway_eip_names" {
  description = "List of Elastic IP names for the NAT gateway."
  value       = local.nat_gateway_eip_names
}

resource "aws_nat_gateway" "gateway" {
  count         = var.subnet_count
  allocation_id = local.nat_gateway_eip_ids[count.index]
  subnet_id     = local.public_subnet_ids[count.index]
  tags = {
    Name = "${local.env}-${local.az_mapping[count.index + 1]}"
  }
}

locals {
  nat_gateway_ids   = aws_nat_gateway.gateway.*.id
  nat_gateway_names = [for i in aws_nat_gateway.gateway.*.tags : i["Name"]]
}

output "nat_gateway_ids" {
  description = "List of NAT gateway IDs."
  value       = local.nat_gateway_ids
}

output "nat_gateway_names" {
  description = "List of NAT gateway names."
  value       = local.nat_gateway_names
}

resource "aws_route_table" "public" {
  vpc_id = local.vpc_id
  tags = {
    Name = "${local.env}-public"
  }

  route {
    cidr_block = "0.0.0.0/0"
    gateway_id = local.internet_gateway_id
  }
}

locals {
  public_route_table_arn  = aws_route_table.public.arn
  public_route_table_id   = aws_route_table.public.id
  public_route_table_name = aws_route_table.public.tags["Name"]
}

output "public_route_table_arn" {
  description = "ARN of the routing table for the public subnets."
  value       = local.public_route_table_arn
}

output "public_route_table_id" {
  description = "ID of the routing table for the public subnets."
  value       = local.public_route_table_id
}

output "public_route_table_name" {
  description = "Name of the routing table for the public subnets."
  value       = local.public_route_table_name
}

resource "aws_route_table_association" "public" {
  for_each       = toset(local.public_subnet_ids)
  route_table_id = local.public_route_table_id
  subnet_id      = each.key
}

locals {
  public_route_table_association_ids = [for i in aws_route_table_association.public : i.id]
}

output "public_route_table_association_ids" {
  description = "List of the route table associations IDs for the public subnets."
  value       = local.public_route_table_association_ids
}

resource "aws_route_table" "private" {
  for_each = toset(local.nat_gateway_ids)
  vpc_id   = local.vpc_id
  tags = {
    Name = "${local.env}-private-${local.az_mapping[index(local.nat_gateway_ids, each.key) + 1]}"
  }

  route {
    cidr_block     = "0.0.0.0/0"
    nat_gateway_id = each.key
  }
}

locals {
  private_route_table_arns  = [for i in aws_route_table.private : i.arn]
  private_route_table_ids   = [for i in aws_route_table.private : i.id]
  private_route_table_names = [for i in aws_route_table.private : i.tags["Name"]]
}

output "private_route_table_arns" {
  description = "List of ARNs of the routing tables for the private subnets."
  value       = local.private_route_table_arns
}

output "private_route_table_ids" {
  description = "List of IDs of the routing tables for the private subnets."
  value       = local.private_route_table_ids
}

output "private_route_table_names" {
  description = "List of names of the routing tables for the private subnets."
  value       = local.private_route_table_names
}

resource "aws_route_table_association" "private" {
  for_each       = zipmap(local.private_subnet_ids, local.private_route_table_ids)
  route_table_id = each.value
  subnet_id      = each.key
}

locals {
  private_route_table_association_ids = [for i in aws_route_table_association.private : i.id]
}

output "private_route_table_association_ids" {
  description = "List of the route table associations IDs for the private subnets."
  value       = local.private_route_table_association_ids
}
