terraform {
  backend "http" {}
  required_providers {
    aws = {
      source  = "hashicorp/aws"
      version = "~> 4.0"
    }
  }
}

locals {
  env     = terraform.workspace == "default" ? "prod" : terraform.workspace
  module  = basename(abspath(path.root))
  name    = "${local.project}-${local.module}-${local.env}"
  project = "homelab"
  common_tags = {
    Environment = local.env
    Module      = local.module
    Name        = local.name
    Project     = local.project
  }
}

output "env" {
  description = "Environment (prod/dev etc.)."
  value       = local.env
}

output "module" {
  description = "The name of the Terraform module, used to tagging resources."
  value       = local.module
}

output "project" {
  description = "The name of the Git project, used to tagging resources."
  value       = local.project
}

variable "region" {
  default     = "us-east-1"
  description = "AWS region."
  type        = string
}

output "region" {
  description = "AWS region."
  value       = var.region
}

provider "aws" {
  region = var.region
  default_tags {
    tags = local.common_tags
  }
}

data "aws_caller_identity" "current" {}

locals {
  account_id = data.aws_caller_identity.current.account_id
}

output "account_id" {
  description = "The AWS account ID."
  value       = local.account_id
}

data "aws_iam_policy_document" "ec2_assume_policy" {
  statement {
    effect  = "Allow"
    actions = ["sts:AssumeRole"]

    principals {
      type        = "Service"
      identifiers = ["ec2.amazonaws.com"]
    }
  }
}

locals {
  ec2_assume_policy = data.aws_iam_policy_document.ec2_assume_policy.json
}

output "ec2_assume_policy" {
  value       = local.ec2_assume_policy
  description = "IAM policy document for EC2 instance assuming a role."
}

data "aws_iam_policy_document" "task_assume_policy" {
  statement {
    effect  = "Allow"
    actions = ["sts:AssumeRole"]

    principals {
      type        = "Service"
      identifiers = ["ecs-tasks.amazonaws.com"]
    }
  }
}

locals {
  task_assume_policy = data.aws_iam_policy_document.task_assume_policy.json
}

output "task_assume_policy" {
  description = "IAM policy document for ECS tasks assuming a role."
  value       = local.task_assume_policy
}

locals {
  resource_group_query = {
    ResourceTypeFilters = [
      "AWS::AllSupported",
    ]
    TagFilters = [
      {
        Key    = "Environment"
        Values = [local.env, ]
      },
      {
        Key    = "Module"
        Values = [local.module, ]
      },
      {
        Key    = "Project"
        Values = [local.project, ]
      },
    ]
  }
}

resource "aws_resourcegroups_group" "group" {
  name = local.name
  resource_query {
    query = jsonencode(local.resource_group_query)
  }
}

locals {
  resource_group_arn  = aws_resourcegroups_group.group.arn
  resource_group_name = aws_resourcegroups_group.group.name
}

output "resource_group_arn" {
  description = "ARN of the resource group."
  value       = local.resource_group_arn
}

output "resource_group_name" {
  description = "Name of the resource group."
  value       = local.resource_group_name
}
