# vim: ft=tf

locals {
  ecs_instance_count = 2
  ecs_instance_type  = "t3.xlarge"
  ecs_key_name       = "Nimrod"
}

data "aws_ec2_instance_type" "ecs" {
  instance_type = local.ecs_instance_type
}

data "aws_ami" "ecs" {
  most_recent = true
  owners      = ["amazon"]

  filter {
    name   = "owner-alias"
    values = ["amazon"]
  }

  filter {
    name   = "name"
    values = ["amzn2-ami-ecs-hvm-*"]
  }

  filter {
    name   = "virtualization-type"
    values = ["hvm"]
  }

  filter {
    name   = "architecture"
    values = data.aws_ec2_instance_type.ecs.supported_architectures
  }
}

resource "aws_ecs_cluster" "this" {
  name = local.name
}

locals {
  long_arn_settings = [
    "serviceLongArnFormat",
    "taskLongArnFormat",
    "containerInstanceLongArnFormat",
  ]
}

resource "aws_ecs_account_setting_default" "this" {
  count = length(local.long_arn_settings)
  name  = element(local.long_arn_settings, count.index)
  value = "enabled"
}

resource "aws_security_group" "ecs" {
  description = "The ${local.name} ECS instances."
  name        = "${local.name}-ecs"
  tags        = { "Name" = "${local.name}-ecs" }
  vpc_id      = aws_vpc.vpc.id

  egress {
    cidr_blocks = ["0.0.0.0/0"]
    from_port   = 0
    protocol    = -1
    to_port     = 0
  }

  ingress {
    cidr_blocks = [var.cidr_block]
    from_port   = 0
    protocol    = -1
    to_port     = 0
  }

  lifecycle {
    create_before_destroy = true
  }
}

locals {
  ecs_instance_policies = [
    "arn:aws:iam::aws:policy/AmazonSSMManagedInstanceCore",
    "arn:aws:iam::aws:policy/service-role/AmazonEC2ContainerServiceforEC2Role",
  ]
}

data "aws_iam_policy_document" "ecs_assume_policy" {
  statement {
    effect  = "Allow"
    actions = ["sts:AssumeRole"]

    principals {
      type        = "Service"
      identifiers = ["ec2.amazonaws.com"]
    }
  }
}

resource "aws_iam_role" "ecs" {
  assume_role_policy = data.aws_iam_policy_document.ecs_assume_policy.json
  description        = "EC2 instances in the ECS cluster in ${local.env}."
  name               = local.name
}

resource "aws_iam_role_policy_attachment" "ecs" {
  for_each   = toset(local.ecs_instance_policies)
  policy_arn = each.key
  role       = aws_iam_role.ecs.name
}

resource "aws_iam_instance_profile" "ecs" {
  name = local.name
  role = aws_iam_role.ecs.name
}

locals {
  user_data_vars = {
    cluster_name = aws_ecs_cluster.this.name
    region       = var.region
  }
  user_data = trimspace(templatefile("${path.module}/user-data.yaml", local.user_data_vars))
}

resource "aws_launch_template" "ecs" {
  image_id                             = data.aws_ami.ecs.image_id
  instance_initiated_shutdown_behavior = "terminate"
  instance_type                        = local.ecs_instance_type
  key_name                             = local.ecs_key_name
  name                                 = local.name
  user_data                            = base64encode(local.user_data)
  vpc_security_group_ids               = [aws_security_group.ecs.id]

  block_device_mappings {
    device_name = "/dev/sda1"

    ebs {
      delete_on_termination = true
      volume_size           = "128"
      volume_type           = "gp3"
    }
  }

  iam_instance_profile {
    arn = aws_iam_instance_profile.ecs.arn
  }
}

resource "aws_autoscaling_group" "ecs" {
  desired_capacity          = local.ecs_instance_count
  force_delete              = true
  health_check_type         = "EC2"
  max_size                  = local.ecs_instance_count
  min_size                  = local.ecs_instance_count
  name                      = local.name
  vpc_zone_identifier       = local.private_subnet_ids
  wait_for_capacity_timeout = "10m"

  instance_refresh {
    strategy = "Rolling"
  }

  launch_template {
    id      = aws_launch_template.ecs.id
    version = aws_launch_template.ecs.latest_version
  }

  dynamic "tag" {
    for_each = local.common_tags

    content {
      key                 = tag.key
      propagate_at_launch = true
      value               = tag.value
    }
  }
}
