resource "aws_cloudwatch_log_group" "lambda" {
  count             = length(local.function_names)
  name              = "/aws/lambda/${local.function_names[count.index]}"
  retention_in_days = var.log_retention
  tags              = local.common_tags
}

locals {
  log_group_arns  = aws_cloudwatch_log_group.lambda.*.arn
  log_group_names = aws_cloudwatch_log_group.lambda.*.name
}

output "log_group_arns" {
  description = "ARNs of the CloudWatch log groups for Lambda function invocations."
  value       = local.log_group_arns
}

output "log_group_names" {
  description = "Names of the CloudWatch log groups for Lambda function invocations."
  value       = local.log_group_names
}

data "aws_iam_policy_document" "log" {
  statement {
    effect = "Allow"

    actions = [
      "logs:CreateLogStream",
      "logs:PutLogEvents",
    ]

    resources = [for arn in local.log_group_arns : "${arn}/*"]
  }
}

locals {
  log_policy_doc = data.aws_iam_policy_document.log.json
}

resource "aws_iam_policy" "log" {
  name   = "${local.module}-${local.env}-log"
  policy = local.log_policy_doc
  tags   = local.common_tags
}

locals {
  log_policy_arn  = aws_iam_policy.log.arn
  log_policy_name = aws_iam_policy.log.name
}

output "log_policy_arn" {
  value       = local.log_policy_arn
  description = "CloudWatch log IAM policy ARN."
}

output "log_policy_name" {
  value       = local.log_policy_name
  description = "CloudWatch log IAM policy name."
}
