locals {
  function_name_prefix = local.Name
  functions = [
    "_dns",
    "gitlab",
  ]
}

data "aws_iam_policy_document" "lambda_assume_policy" {
  statement {
    effect  = "Allow"
    actions = ["sts:AssumeRole"]

    principals {
      type        = "Service"
      identifiers = ["lambda.amazonaws.com"]
    }
  }
}

locals {
  lambda_assume_policy_doc = data.aws_iam_policy_document.lambda_assume_policy.json
}

resource "aws_iam_role" "lambda" {
  name               = local.Name
  assume_role_policy = local.lambda_assume_policy_doc
  tags               = local.common_tags
}

locals {
  lambda_role_arn  = aws_iam_role.lambda.arn
  lambda_role_name = aws_iam_role.lambda.name
}

output "lambda_role_arn" {
  description = "ARN of the Lambda function IAM role."
  value       = local.lambda_role_arn
}

output "lambda_role_name" {
  description = "Name of the Lambda function IAM role."
  value       = local.lambda_role_name
}

locals {
  function_role_policies = [
    local.publish_policy_arn,
    #local.log_policy_arn,
    "arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole",
  ]
}

resource "aws_iam_role_policy_attachment" "function" {
  count      = length(local.function_role_policies)
  role       = local.lambda_role_name
  policy_arn = local.function_role_policies[count.index]
}

variable "runtime" {
  default     = "python3.8"
  description = "The Lambda function runtime."
  type        = string
}

output "runtime" {
  description = "The Lambda function runtime."
  value       = var.runtime
}

variable "memory_size" {
  default     = 128
  description = "Amount of memory in MB allocated to the function invocation."
  type        = number
}

output "memory_size" {
  description = "Amount of memory in MB allocated to the function invocation."
  value       = var.memory_size
}

variable "timeout" {
  default     = 30
  description = "Amouyynt of time in seconds the function has to run."
  type        = number
}

output "timeout" {
  description = "Amount of memory in MB allocated to the function invocation."
  value       = var.timeout
}

resource "aws_lambda_function" "function" {
  count             = length(local.functions)
  runtime           = var.runtime
  function_name     = "${local.function_name_prefix}-${replace(local.functions[count.index], "_", "")}"
  role              = local.lambda_role_arn
  source_code_hash  = filebase64sha256("payload.zip")
  s3_bucket         = local.payloads_bucket_name
  s3_key            = local.payload_object_name
  s3_object_version = local.payload_object_version
  package_type      = "Zip"
  handler           = "${local.functions[count.index]}.handler"
  description       = "${local.module} ${local.functions[count.index]} check in ${local.env}."
  memory_size       = var.memory_size
  tags              = local.common_tags
  timeout           = var.timeout

  environment {
    variables = {
      ENV       = local.env
      MODULE    = local.module
      TOPIC_ARN = local.topic_arn
      VERSION   = local.payload_object_version
    }
  }

  # Create the log group with retention before the function is created.
  # Otherwise it's created without retention and need to be imported.
  depends_on = [
    aws_cloudwatch_log_group.lambda,
  ]
}

locals {
  function_arns     = aws_lambda_function.function.*.arn
  function_names    = aws_lambda_function.function.*.id
  function_versions = aws_lambda_function.function.*.version
}

output "function_arns" {
  description = "ARNs of the Lambda functions."
  value       = local.function_arns
}

output "function_names" {
  description = "Names of the Lambda functions."
  value       = local.function_names
}

output "function_versions" {
  description = "Versions of the Lambda functions."
  value       = local.function_versions
}

output "function_version_mapping" {
  description = "Mapping of the function and version."
  value       = zipmap(local.functions, local.function_versions)
}

resource "aws_lambda_alias" "function" {
  count            = length(local.functions)
  name             = "${local.function_name_prefix}_${local.functions[count.index]}"
  function_name    = local.function_arns[count.index]
  function_version = local.function_versions[count.index]
}

locals {
  function_alias_arns  = aws_lambda_alias.function.*.arn
  function_alias_names = aws_lambda_alias.function.*.name
}

output "function_alias_arns" {
  description = "ARNs of the Lambda functions aliases."
  value       = local.function_alias_arns
}

output "function_alias_names" {
  description = "Names of the Lambda functions aliases."
  value       = local.function_alias_names
}

resource "aws_lambda_permission" "cloudwatch" {
  count         = length(local.function_arns)
  statement_id  = "AllowExecutionFromCloudWatch"
  action        = "lambda:InvokeFunction"
  principal     = "events.amazonaws.com"
  source_arn    = local.cloudwatch_rule_arn
  function_name = local.function_names[count.index]
  qualifier     = local.function_alias_names[count.index]
}
